<?php
/**
 * Date and Time Locale object
 *
 * @package WordPress
 * @subpackage i18n
 */

/**
 * Class that loads the calendar locale.
 *
 * @since 2.1.0
 */
class WP_Locale {
	/**
	 * Stores the translated strings for the full weekday names.
	 *
	 * @since 2.1.0
	 * @var array
	 * @access private
	 */
	var $weekday;

	/**
	 * Stores the translated strings for the one character weekday names.
	 *
	 * There is a hack to make sure that tiistai and torstai, as well
	 * as sunnuntai and lauantai don't conflict. See init() method for more.
	 *
	 * @see WP_Locale::init() for how to handle the hack.
	 *
	 * @since 2.1.0
	 * @var array
	 * @access private
	 */
	var $weekday_initial;

	/**
	 * Stores the translated strings for the abbreviated weekday names.
	 *
	 * @since 2.1.0
	 * @var array
	 * @access private
	 */
	var $weekday_abbrev;

	/**
	 * Stores the translated strings for the full month names.
	 *
	 * @since 2.1.0
	 * @var array
	 * @access private
	 */
	var $month;

	/**
	 * Stores the translated strings for the abbreviated month names.
	 *
	 * @since 2.1.0
	 * @var array
	 * @access private
	 */
	var $month_abbrev;

	/**
	 * Stores the translated strings for 'am' and 'pm'.
	 *
	 * Also the capalized versions.
	 *
	 * @since 2.1.0
	 * @var array
	 * @access private
	 */
	var $meridiem;

	/**
	 * The text direction of the locale language.
	 *
	 * Default is left to right 'ltr'.
	 *
	 * @since 2.1.0
	 * @var string
	 * @access private
	 */
	var $text_direction = 'ltr';

	/**
	 * Imports the global version to the class property.
	 *
	 * @since 2.1.0
	 * @var array
	 * @access private
	 */
	var $locale_vars = array('text_direction');

	/**
	 * Sets up the translated strings and object properties.
	 *
	 * The method creates the translatable strings for various
	 * calendar elements. Which allows for specifying locale
	 * specific calendar names and text direction.
	 *
	 * @since 2.1.0
	 * @access private
	 */
	function init() {
		// The Weekdays
		$this->weekday[0] = __('sunnuntai');
		$this->weekday[1] = __('maanantai');
		$this->weekday[2] = __('tiistai');
		$this->weekday[3] = __('keskiviikko');
		$this->weekday[4] = __('torstai');
		$this->weekday[5] = __('perjantai');
		$this->weekday[6] = __('lauantai');

		// The first letter of each day.  The _%day%_initial suffix is a hack to make
		// sure the day initials are unique.
		$this->weekday_initial[__('sunnuntai')]    = __('S_sunnuntai_initial');
		$this->weekday_initial[__('maanantai')]    = __('M_maanantai_initial');
		$this->weekday_initial[__('tiistai')]   = __('T_tiistai_initial');
		$this->weekday_initial[__('keskiviikko')] = __('K_keskiviikko_initial');
		$this->weekday_initial[__('torstai')]  = __('T_torstai_initial');
		$this->weekday_initial[__('perjantai')]    = __('P_perjantai_initial');
		$this->weekday_initial[__('lauantai')]  = __('L_lauantai_initial');

		foreach ($this->weekday_initial as $weekday_ => $weekday_initial_) {
			$this->weekday_initial[$weekday_] = preg_replace('/_.+_initial$/', '', $weekday_initial_);
		}

		// Abbreviations for each day.
		$this->weekday_abbrev[__('sunnuntai')]    = __('su');
		$this->weekday_abbrev[__('maanantai')]    = __('ma');
		$this->weekday_abbrev[__('tiistai')]   = __('ti');
		$this->weekday_abbrev[__('keskiviikko')] = __('ke');
		$this->weekday_abbrev[__('torstai')]  = __('to');
		$this->weekday_abbrev[__('perjantai')]    = __('pe');
		$this->weekday_abbrev[__('lauantai')]  = __('la');

		// The Months
		$this->month['01'] = __('tammikuu');
		$this->month['02'] = __('helmikuu');
		$this->month['03'] = __('maaliskuu');
		$this->month['04'] = __('huhtikuu');
		$this->month['05'] = __('toukokuu');
		$this->month['06'] = __('kesäkuu');
		$this->month['07'] = __('heinäkuu');
		$this->month['08'] = __('elokuu');
		$this->month['09'] = __('syyskuu');
		$this->month['10'] = __('lokakuu');
		$this->month['11'] = __('marraskuu');
		$this->month['12'] = __('joulukuu');

		// Abbreviations for each month. Uses the same hack as above to get around the
		// 'toukokuu' duplication.
		$this->month_abbrev[__('tammikuu')] = __('tamm_tammikuu_abbreviation');
		$this->month_abbrev[__('helmikuu')] = __('hel_helmikuu_abbreviation');
		$this->month_abbrev[__('maaliskuu')] = __('maa_maaliskuu_abbreviation');
		$this->month_abbrev[__('huhtikuu')] = __('huh_huhtikuu_abbreviation');
		$this->month_abbrev[__('toukokuu')] = __('tou_toukokuu_abbreviation');
		$this->month_abbrev[__('kesäkuu')] = __('kes_kesäkuu_abbreviation');
		$this->month_abbrev[__('heinäkuu')] = __('hei_heinäkuu_abbreviation');
		$this->month_abbrev[__('elokuu')] = __('elo_elokuu_abbreviation');
		$this->month_abbrev[__('syyskuu')] = __('syy_syyskuu_abbreviation');
		$this->month_abbrev[__('lokakuu')] = __('lok_lokakuu_abbreviation');
		$this->month_abbrev[__('marraskuu')] = __('mar_marraskuu_abbreviation');
		$this->month_abbrev[__('joulukuu')] = __('jou_joulukuu_abbreviation');

		foreach ($this->month_abbrev as $month_ => $month_abbrev_) {
			$this->month_abbrev[$month_] = preg_replace('/_.+_abbreviation$/', '', $month_abbrev_);
		}

		// The Meridiems
		$this->meridiem['am'] = __('am');
		$this->meridiem['pm'] = __('pm');
		$this->meridiem['AM'] = __('AM');
		$this->meridiem['PM'] = __('PM');

		// Numbers formatting
		// See http://php.net/number_format

		$trans = _c('number_format_decimals|$decimals argument for http://php.net/number_format, default is 0');
		$this->number_format['decimals'] = ('number_format_decimals' == $trans) ? 0 : $trans;

		$trans = _c('number_format_decimal_point|$dec_point argument for http://php.net/number_format, default is .');
		$this->number_format['decimal_point'] = ('number_format_decimal_point' == $trans) ? '.' : $trans;

		$trans = _c('number_format_thousands_sep|$thousands_sep argument for http://php.net/number_format, default is ,');
		$this->number_format['thousands_sep'] = ('number_format_thousands_sep' == $trans) ? ',' : $trans;

		// Import global locale vars set during inclusion of $locale.php.
		foreach ( (array) $this->locale_vars as $var ) {
			if ( isset($GLOBALS[$var]) )
				$this->$var = $GLOBALS[$var];
		}

	}

	/**
	 * Retrieve the full translated weekday word.
	 *
	 * Week starts on translated sunnuntai and can be fetched
	 * by using 0 (zero). So the week starts with 0 (zero)
	 * and ends on lauantai with is fetched by using 6 (six).
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @param int $weekday_number 0 for sunnuntai through 6 lauantai
	 * @return string Full translated weekday
	 */
	function get_weekday($weekday_number) {
		return $this->weekday[$weekday_number];
	}

	/**
	 * Retrieve the translated weekday initial.
	 *
	 * The weekday initial is retrieved by the translated
	 * full weekday word. When translating the weekday initial
	 * pay attention to make sure that the starting letter does
	 * not conflict.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @param string $weekday_name
	 * @return string
	 */
	function get_weekday_initial($weekday_name) {
		return $this->weekday_initial[$weekday_name];
	}

	/**
	 * Retrieve the translated weekday abbreviation.
	 *
	 * The weekday abbreviation is retrieved by the translated
	 * full weekday word.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @param string $weekday_name Full translated weekday word
	 * @return string Translated weekday abbreviation
	 */
	function get_weekday_abbrev($weekday_name) {
		return $this->weekday_abbrev[$weekday_name];
	}

	/**
	 * Retrieve the full translated month by month number.
	 *
	 * The $month_number parameter has to be a string
	 * because it must have the '0' in front of any number
	 * that is less than 10. Starts from '01' and ends at
	 * '12'.
	 *
	 * You can use an integer instead and it will add the
	 * '0' before the numbers less than 10 for you.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @param string|int $month_number '01' through '12'
	 * @return string Translated full month name
	 */
	function get_month($month_number) {
		return $this->month[zeroise($month_number, 2)];
	}

	/**
	 * Retrieve translated version of month abbreviation string.
	 *
	 * The $month_name parameter is expected to be the translated or
	 * translatable version of the month.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @param string $month_name Translated month to get abbreviated version
	 * @return string Translated abbreviated month
	 */
	function get_month_abbrev($month_name) {
		return $this->month_abbrev[$month_name];
	}

	/**
	 * Retrieve translated version of meridiem string.
	 *
	 * The $meridiem parameter is expected to not be translated.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @param string $meridiem Either 'am', 'pm', 'AM', or 'PM'. Not translated version.
	 * @return string Translated version
	 */
	function get_meridiem($meridiem) {
		return $this->meridiem[$meridiem];
	}

	/**
	 * Global variables are deprecated. For backwards compatibility only.
	 *
	 * @deprecated For backwards compatibility only.
	 * @access private
	 *
	 * @since 2.1.0
	 */
	function register_globals() {
		$GLOBALS['weekday']         = $this->weekday;
		$GLOBALS['weekday_initial'] = $this->weekday_initial;
		$GLOBALS['weekday_abbrev']  = $this->weekday_abbrev;
		$GLOBALS['month']           = $this->month;
		$GLOBALS['month_abbrev']    = $this->month_abbrev;
	}

	/**
	 * PHP4 style constructor which calls helper methods to set up object variables
	 *
	 * @uses WP_Locale::init()
	 * @uses WP_Locale::register_globals()
	 * @since 2.1.0
	 *
	 * @return WP_Locale
	 */
	function WP_Locale() {
		$this->init();
		$this->register_globals();
	}
}

?>